<?php
/**
 * Dynamic OpenGraph Image Generator for Syamsu Files
 * Generates a 1200x630 PNG image for shared file links
 * Usage: /og-image.php?name=filename.pdf&size=2.5+MB&type=pdf
 */

// Params
$fileName = $_GET['name'] ?? 'File';
$fileSize = $_GET['size'] ?? '';
$fileType = strtolower($_GET['type'] ?? 'file');
$appName  = 'Syamsu Files';

// Sanitize
$fileName = htmlspecialchars(strip_tags($fileName), ENT_QUOTES, 'UTF-8');
$fileSize = htmlspecialchars(strip_tags($fileSize), ENT_QUOTES, 'UTF-8');

// Emoji/icon by type
$emoji = '📎';
$color = '#9b6dd7';
if (str_starts_with($fileType, 'image')) { $emoji = '🖼️'; $color = '#e879a8'; }
elseif ($fileType === 'pdf') { $emoji = '📕'; $color = '#d94f5c'; }
elseif (str_starts_with($fileType, 'video')) { $emoji = '🎬'; $color = '#6fa8dc'; }
elseif (str_starts_with($fileType, 'audio')) { $emoji = '🎵'; $color = '#5bbcb4'; }
elseif (in_array($fileType, ['zip', 'rar', '7z'])) { $emoji = '📦'; $color = '#f0c24a'; }
elseif (str_starts_with($fileType, 'text')) { $emoji = '📝'; $color = '#9b6dd7'; }

// Truncate long filename
if (mb_strlen($fileName) > 40) {
    $fileName = mb_substr($fileName, 0, 37) . '...';
}

// Check if GD is available
if (!function_exists('imagecreatetruecolor')) {
    // Fallback: serve a static SVG as image
    header('Content-Type: image/svg+xml');
    echo <<<SVG
<svg xmlns="http://www.w3.org/2000/svg" width="1200" height="630" viewBox="0 0 1200 630">
  <defs>
    <linearGradient id="bg" x1="0%" y1="0%" x2="100%" y2="100%">
      <stop offset="0%" style="stop-color:#f0e6ff"/>
      <stop offset="100%" style="stop-color:#fce4ec"/>
    </linearGradient>
  </defs>
  <rect width="1200" height="630" fill="url(#bg)"/>
  <!-- Grid pattern -->
  <pattern id="grid" width="30" height="30" patternUnits="userSpaceOnUse">
    <path d="M 30 0 L 0 0 0 30" fill="none" stroke="rgba(180,160,210,0.2)" stroke-width="1"/>
  </pattern>
  <rect width="1200" height="630" fill="url(#grid)"/>
  <!-- Card -->
  <rect x="200" y="115" width="800" height="400" rx="20" fill="#fffbfe" stroke="#7c6b9e" stroke-width="3"/>
  <!-- Card header -->
  <rect x="200" y="115" width="800" height="56" rx="20" fill="{$color}"/>
  <rect x="200" y="151" width="800" height="20" fill="{$color}"/>
  <!-- Dots -->
  <circle cx="228" cy="143" r="8" fill="#ff6b6b" stroke="rgba(0,0,0,0.2)" stroke-width="1.5"/>
  <circle cx="252" cy="143" r="8" fill="#ffd93d" stroke="rgba(0,0,0,0.2)" stroke-width="1.5"/>
  <circle cx="276" cy="143" r="8" fill="#6bcb77" stroke="rgba(0,0,0,0.2)" stroke-width="1.5"/>
  <!-- Title text -->
  <text x="300" y="150" font-family="system-ui,sans-serif" font-size="18" font-weight="bold" fill="#ffffff">📁 {$appName} — Unduh File</text>
  <!-- File icon area -->
  <text x="600" y="310" font-family="system-ui,sans-serif" font-size="72" text-anchor="middle">{$emoji}</text>
  <!-- File name -->
  <text x="600" y="380" font-family="system-ui,sans-serif" font-size="28" font-weight="bold" fill="#3d2c5e" text-anchor="middle">{$fileName}</text>
  <!-- File size -->
  <text x="600" y="415" font-family="system-ui,sans-serif" font-size="18" fill="#9a8bb5" text-anchor="middle">{$fileSize}</text>
  <!-- Download CTA -->
  <rect x="430" y="440" width="340" height="50" rx="12" fill="{$color}"/>
  <text x="600" y="472" font-family="system-ui,sans-serif" font-size="20" font-weight="bold" fill="#ffffff" text-anchor="middle">💾 Unduh File</text>
  <!-- Sparkles -->
  <text x="180" y="160" font-size="24" opacity="0.15">✦</text>
  <text x="980" y="520" font-size="20" opacity="0.15">✧</text>
  <!-- Footer -->
  <text x="600" y="560" font-family="system-ui,sans-serif" font-size="14" fill="#9a8bb5" text-anchor="middle">✦ Dibagikan melalui {$appName} ✦</text>
</svg>
SVG;
    exit;
}

// === GD-based PNG generation ===
$width = 1200;
$height = 630;
$img = imagecreatetruecolor($width, $height);
imagesavealpha($img, true);

// Colors
$bgColor1 = imagecolorallocate($img, 240, 230, 255);
$bgColor2 = imagecolorallocate($img, 252, 228, 236);
$panelColor = imagecolorallocate($img, 255, 251, 254);
$borderColor = imagecolorallocate($img, 124, 107, 158);
$inkColor = imagecolorallocate($img, 61, 44, 94);
$mutedColor = imagecolorallocate($img, 154, 139, 181);
$white = imagecolorallocate($img, 255, 255, 255);
$gridColor = imagecolorallocate($img, 210, 195, 235);

// Background gradient (simple vertical)
for ($y = 0; $y < $height; $y++) {
    $ratio = $y / $height;
    $r = (int)(240 + (252 - 240) * $ratio);
    $g = (int)(230 + (228 - 230) * $ratio);
    $b = (int)(255 + (236 - 255) * $ratio);
    $lineColor = imagecolorallocate($img, $r, $g, $b);
    imageline($img, 0, $y, $width, $y, $lineColor);
}

// Grid pattern
for ($x = 0; $x < $width; $x += 30) {
    imageline($img, $x, 0, $x, $height, $gridColor);
}
for ($y = 0; $y < $height; $y += 30) {
    imageline($img, 0, $y, $width, $y, $gridColor);
}

// Card background
imagefilledroundedrect($img, 200, 115, 1000, 515, 16, $panelColor);
imageroundedrect($img, 200, 115, 1000, 515, 16, $borderColor);

// Header bar
$headerR = hexdec(substr($color, 1, 2));
$headerG = hexdec(substr($color, 3, 2));
$headerB = hexdec(substr($color, 5, 2));
$headerColor = imagecolorallocate($img, $headerR, $headerG, $headerB);
imagefilledroundedrect($img, 200, 115, 1000, 170, 16, $headerColor);
imagefilledrectangle($img, 200, 145, 1000, 170, $headerColor);

// Dots
$dotRed = imagecolorallocate($img, 255, 107, 107);
$dotYellow = imagecolorallocate($img, 255, 217, 61);
$dotGreen = imagecolorallocate($img, 107, 203, 119);
imagefilledellipse($img, 228, 143, 16, 16, $dotRed);
imagefilledellipse($img, 252, 143, 16, 16, $dotYellow);
imagefilledellipse($img, 276, 143, 16, 16, $dotGreen);

// Title text
$font = null; // Use default
imagestring($img, 5, 300, 135, "$appName - Unduh File", $white);

// File name
$nameDisplay = $fileName;
$nameWidth = strlen($nameDisplay) * imagefontwidth(5);
$nameX = (int)(($width - $nameWidth) / 2);
imagestring($img, 5, $nameX, 340, $nameDisplay, $inkColor);

// File size
if ($fileSize) {
    $sizeWidth = strlen($fileSize) * imagefontwidth(4);
    $sizeX = (int)(($width - $sizeWidth) / 2);
    imagestring($img, 4, $sizeX, 370, $fileSize, $mutedColor);
}

// Download button
imagefilledroundedrect($img, 430, 410, 770, 460, 12, $headerColor);
$dlText = "Unduh File";
$dlWidth = strlen($dlText) * imagefontwidth(5);
$dlX = (int)(($width - $dlWidth) / 2);
imagestring($img, 5, $dlX, 428, $dlText, $white);

// Footer
$footerText = "Dibagikan melalui $appName";
$footerWidth = strlen($footerText) * imagefontwidth(3);
$footerX = (int)(($width - $footerWidth) / 2);
imagestring($img, 3, $footerX, 490, $footerText, $mutedColor);

// Output
header('Content-Type: image/png');
header('Cache-Control: public, max-age=86400');
imagepng($img);
imagedestroy($img);

// Helper functions for rounded rectangles
function imagefilledroundedrect($img, $x1, $y1, $x2, $y2, $r, $color) {
    imagefilledrectangle($img, $x1 + $r, $y1, $x2 - $r, $y2, $color);
    imagefilledrectangle($img, $x1, $y1 + $r, $x2, $y2 - $r, $color);
    imagefilledellipse($img, $x1 + $r, $y1 + $r, $r * 2, $r * 2, $color);
    imagefilledellipse($img, $x2 - $r, $y1 + $r, $r * 2, $r * 2, $color);
    imagefilledellipse($img, $x1 + $r, $y2 - $r, $r * 2, $r * 2, $color);
    imagefilledellipse($img, $x2 - $r, $y2 - $r, $r * 2, $r * 2, $color);
}

function imageroundedrect($img, $x1, $y1, $x2, $y2, $r, $color) {
    imageline($img, $x1 + $r, $y1, $x2 - $r, $y1, $color);
    imageline($img, $x1 + $r, $y2, $x2 - $r, $y2, $color);
    imageline($img, $x1, $y1 + $r, $x1, $y2 - $r, $color);
    imageline($img, $x2, $y1 + $r, $x2, $y2 - $r, $color);
    imagearc($img, $x1 + $r, $y1 + $r, $r * 2, $r * 2, 180, 270, $color);
    imagearc($img, $x2 - $r, $y1 + $r, $r * 2, $r * 2, 270, 360, $color);
    imagearc($img, $x1 + $r, $y2 - $r, $r * 2, $r * 2, 90, 180, $color);
    imagearc($img, $x2 - $r, $y2 - $r, $r * 2, $r * 2, 0, 90, $color);
}
